<?php
/**
 * Classe para gerenciar a instalação e atualização de plugins.
 * @version 1.8 - Corrigido erro de permissão no move, usando copy_dir.
 */
class PluginUpdater {

    /**
     * @var string A URL do pacote do plugin a ser baixado.
     */
    private $package_url;

    /**
     * Atualiza um plugin manualmente: baixa, descompacta, substitui e reativa.
     *
     * @param string $plugin_file_path O caminho do arquivo principal do plugin (ex: meu-plugin/meu-plugin.php).
     * @param string $package_url A URL para o arquivo .zip do plugin.
     * @return bool|WP_Error Retorna true em sucesso, ou um objeto WP_Error em caso de falha.
     */
    public function update_plugin($plugin_file_path, $package_url) {
        error_log('[MRP Updater V5] Iniciando processo de atualização MANUAL.');
        error_log('[MRP Updater V5] Plugin alvo: ' . $plugin_file_path);
        error_log('[MRP Updater V5] URL do pacote: ' . $package_url);

        require_once ABSPATH . 'wp-admin/includes/file.php';
        require_once ABSPATH . 'wp-admin/includes/plugin.php';
        // É necessário garantir que as funções de cópia de diretório estejam disponíveis
        if (!function_exists('copy_dir')) {
            require_once ABSPATH . 'wp-admin/includes/upgrade.php'; // Inclui funções auxiliares como copy_dir
        }

        WP_Filesystem();
        global $wp_filesystem;

        $plugins_dir = $wp_filesystem->wp_plugins_dir();
        if (!$plugins_dir) {
            return new WP_Error('no_plugins_dir', 'Não foi possível determinar o diretório de plugins.');
        }

        // $plugin_path = dirname($plugins_dir . $plugin_file_path); // REMOVIDO: Calculado dinamicamente agora.
        
        $is_active = is_plugin_active($plugin_file_path);
        if ($is_active) {
            deactivate_plugins($plugin_file_path, true);
            error_log('[MRP Updater V5] Plugin desativado com sucesso.');
        }

        // --- Etapa de Download Melhorada ---
        error_log('[MRP Updater V5] Baixando pacote com wp_remote_get e sslverify=false.');
        $response = wp_remote_get($package_url, [
            'timeout'   => 300,
            'sslverify' => false // Força a desativação da verificação SSL
        ]);

        if (is_wp_error($response)) {
            error_log('[MRP Updater V5] ERRO em wp_remote_get: ' . $response->get_error_message());
            return $response;
        }

        $response_code = wp_remote_retrieve_response_code($response);
        if ($response_code !== 200) {
            error_log('[MRP Updater V5] ERRO: O servidor retornou o código de status HTTP ' . $response_code);
            
            // Log headers to debug 429 errors
            $headers = wp_remote_retrieve_headers($response);
            error_log('[MRP Updater V5] Headers da resposta de erro: ' . print_r($headers, true));
            
            return new WP_Error('download_failed_http', 'Falha no download. O servidor respondeu com o código: ' . $response_code);
        }

        // Salva o conteúdo em um arquivo temporário
        $package_data = wp_remote_retrieve_body($response);
        $temp_file = get_temp_dir() . wp_unique_filename(get_temp_dir(), basename($package_url));
        $write_result = $wp_filesystem->put_contents($temp_file, $package_data);

        if (!$write_result) {
            error_log('[MRP Updater V5] ERRO: Não foi possível salvar o pacote baixado em um arquivo temporário.');
            return new WP_Error('write_temp_failed', 'Falha ao salvar o pacote em um arquivo temporário.');
        }
        error_log('[MRP Updater V5] Pacote baixado e salvo com sucesso em: ' . $temp_file);
        
        // DEBUG: Verificar cabeçalho do arquivo
        $file_header = file_get_contents($temp_file, false, null, 0, 50);
        error_log('[MRP Updater V5] Primeiros 50 bytes do arquivo: ' . bin2hex($file_header));
        error_log('[MRP Updater V5] Conteúdo texto (parcial): ' . substr($file_header, 0, 50));
        
        if (strpos($file_header, 'PK') !== 0) {
            error_log('[MRP Updater V5] AVISO: O arquivo baixado não parece ser um ZIP válido (não começa com PK).');
        }
        // --- Fim da Etapa de Download Melhorada ---

        $temp_unzip_dir = get_temp_dir() . 'mrp_unzip_' . time();
        error_log('[MRP Updater V5] Descompactando para: ' . $temp_unzip_dir);
        $unzip_result = unzip_file($temp_file, $temp_unzip_dir);
        $wp_filesystem->delete($temp_file);

        if (is_wp_error($unzip_result)) {
            error_log('[MRP Updater V5] ERRO ao descompactar: ' . $unzip_result->get_error_message());
            return $unzip_result;
        }

        $unzipped_folders = $wp_filesystem->dirlist($temp_unzip_dir);
        if (!$unzipped_folders || !is_array($unzipped_folders)) {
             return new WP_Error('unzip_failed', 'Falha ao listar o conteúdo do diretório descompactado.');
        }
        $source_dir_name = key($unzipped_folders);
        $source_path = trailingslashit($temp_unzip_dir) . $source_dir_name;
        error_log('[MRP Updater V5] Caminho de origem descompactado: ' . $source_path);

        // --- MODIFICAÇÃO: Determinar o caminho de destino baseado no zip ---
        // Isso permite instalação (quando não temos plugin_path) e atualização segura.
        $target_plugin_path = trailingslashit($plugins_dir) . $source_dir_name;
        error_log('[MRP Updater V5] Caminho de destino final: ' . $target_plugin_path);

        // Se o diretório de destino já existe, removemos (Atualização/Sobrescrita)
        if ($wp_filesystem->exists($target_plugin_path)) {
            error_log('[MRP Updater V5] O diretório de destino já existe. Removendo para substituir...');
            $delete_result = $wp_filesystem->delete($target_plugin_path, true);
            if (!$delete_result) {
                error_log('[MRP Updater V5] ERRO: Falha ao excluir a pasta antiga. Verifique as permissões.');
                $wp_filesystem->delete($temp_unzip_dir, true);
                return new WP_Error('delete_failed', 'Não foi possível remover a versão antiga do plugin.');
            }
        }
        
        // Copia a nova versão
        error_log('[MRP Updater V5] Copiando arquivos para: ' . $target_plugin_path);
        $copy_result = copy_dir($source_path, $target_plugin_path);

        // Limpa o diretório descompactado temporário
        $wp_filesystem->delete($temp_unzip_dir, true);
        
        if (is_wp_error($copy_result) || !$copy_result) {
            error_log('[MRP Updater V5] ERRO: Falha ao copiar a nova versão. ' . (is_wp_error($copy_result) ? $copy_result->get_error_message() : ''));
            return new WP_Error('copy_failed', 'Não foi possível instalar a nova versão do plugin. Erro de cópia.');
        }
        error_log('[MRP Updater V5] Instalação/Atualização concluída com sucesso.');
        // --- FIM DA MODIFICAÇÃO ---


        if ($is_active) {
            activate_plugin($plugin_file_path);
            error_log('[MRP Updater V5] Plugin reativado.');
        }
        
        clearstatcache();
        wp_cache_delete('plugins', 'plugins');
        $new_plugin_data = get_plugin_data($plugins_dir . $plugin_file_path);
        error_log('[MRP Updater V5] VERIFICAÇÃO FINAL: Versão encontrada no disco: ' . $new_plugin_data['Version']);

        return true;
    }
}