<?php
/*
  Plugin Name: Meu Repositório Client
  Description: Cliente para buscar e atualizar plugins de um repositório privado via REST API com suporte a cache, rate limiting e estatísticas.
  Version: 4.0
  TAG: true
  Author: Thomas Marcelino
  Author URI: https://wpmasters.com.br
  License: GPL2
  Text Domain: meu-repositorio-client
*/

defined('ABSPATH') or die('Acesso direto não permitido!');

// --- Constantes e Inclusões ---
define('MRP_PLUGIN_PATH', plugin_dir_path(__FILE__));
define('MRP_PLUGIN_URL', plugin_dir_url(__FILE__));
// Novo Cache Buster
define('MRP_CACHE_BUSTER', time());

require_once MRP_PLUGIN_PATH . 'includes/class-folder-scanner.php';
require_once MRP_PLUGIN_PATH . 'includes/admin-settings-page.php';
require_once MRP_PLUGIN_PATH . 'includes/class-plugin-updater.php';
require_once MRP_PLUGIN_PATH . 'includes/class-api-cache.php';

/**
 * Classe principal e final do plugin.
 * @version 3.2
 */
final class MeuRepositorioPlugin {

    private static $instance;
    private $api_url;
    private $plugin_version;
    private $settings_page_hook;

    /**
     * O construtor é privado para garantir que a classe seja instanciada
     * apenas uma vez (padrão Singleton).
     */
    private function __construct() {
        // A lógica principal agora é acionada no gancho 'init'.
        add_action('init', [$this, 'init']);
    }

    /**
     * Ponto de entrada para obter a instância da classe.
     *
     * @return MeuRepositorioPlugin A instância única da classe.
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Inicializa o plugin, registra ganchos e carrega propriedades.
     * É executado no gancho 'init' para garantir o carregamento correto.
     */
    public function init() {
        // Carrega o text domain no momento correto.
        load_plugin_textdomain('meu-repositorio-client', false, dirname(plugin_basename(__FILE__)) . '/languages/');

        // Carrega as propriedades necessárias.
        $this->api_url = get_option('meu_repositorio_api_url', '');
        if (!function_exists('get_plugin_data')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        // Lê a versão do cabeçalho do plugin principal
        $this->plugin_version = get_plugin_data(__FILE__)['Version'] ?? '3.2';

        // Registra ganchos que dependem de um contexto (admin, ajax, etc).
        if (is_admin()) {
            add_action('admin_menu', [$this, 'mrp_add_admin_menu']);
            add_action('admin_enqueue_scripts', [$this, 'mrp_enqueue_admin_assets']);
            add_action('wp_ajax_meu_repositorio_update_plugin', [$this, 'mrp_handle_update_plugin']);
            add_action('wp_ajax_meu_repositorio_activate_plugin', [$this, 'mrp_handle_activate_plugin']);
            add_action('wp_ajax_meu_repositorio_force_update', [$this, 'mrp_force_update_plugins_ajax_handler']);
        }

        if (!empty($this->api_url)) {
            add_filter('pre_set_site_transient_update_plugins', [$this, 'mrp_check_for_updates']);
        }
    }

    /**
     * Adiciona o menu do plugin ao painel de administração.
     */
    public function mrp_add_admin_menu() {
        $update_count = $this->mrp_get_update_count();
        $menu_title = __('Repositório', 'meu-repositorio-client');

        if ($update_count > 0) {
            $bubble = " <span class='update-plugins count-{$update_count}'><span class='plugin-count'>{$update_count}</span></span>";
            $menu_title .= $bubble;
        }

        add_menu_page(__('Repositório de Plugins', 'meu-repositorio-client'), $menu_title, 'manage_options', 'meu-repositorio-client', [$this, 'mrp_render_repo_page'], 'dashicons-download', 81);
        add_submenu_page('meu-repositorio-client', __('Plugins do Repositório', 'meu-repositorio-client'), __('Plugins do Repositório', 'meu-repositorio-client'), 'manage_options', 'meu-repositorio-client', [$this, 'mrp_render_repo_page']);
        $this->settings_page_hook = add_submenu_page('meu-repositorio-client', __('Configurações', 'meu-repositorio-client'), __('Configurações', 'meu-repositorio-client'), 'manage_options', 'meu-repositorio-settings', 'mrp_render_settings_page');
    }

    /**
     * Calcula e retorna o número de atualizações de plugins disponíveis.
     */
    private function mrp_get_update_count() {
        $update_count = get_transient('mrp_update_count');
        if (false !== $update_count) {
            return (int) $update_count;
        }

        $count = 0;
        if (!empty($this->api_url)) {
            $folder_scanner = new MRP_FolderScanner($this->api_url);
            $available_plugins = $folder_scanner->scan_for_plugins();

            if (!is_wp_error($available_plugins) && !empty($available_plugins)) {
                $installed_plugins = get_plugins();
                foreach ($available_plugins as $plugin_slug => $plugin_data) {
                    foreach ($installed_plugins as $path => $details) {
                        if (strpos($path, $plugin_slug . '/') === 0 || $details['Name'] === $plugin_data['name']) {
                            if (version_compare($plugin_data['version'], $details['Version'], '>')) {
                                $count++;
                            }
                            break;
                        }
                    }
                }
            }
        }

        set_transient('mrp_update_count', $count, HOUR_IN_SECONDS);
        return $count;
    }

    /**
     * Enfileira os scripts e estilos.
     */
    public function mrp_enqueue_admin_assets($hook) {
        $pages = ['toplevel_page_meu-repositorio-client', $this->settings_page_hook];
        if (!in_array($hook, $pages)) {
            return;
        }

        wp_enqueue_style('meu-repositorio-admin-style', MRP_PLUGIN_URL . 'assets/css/admin-style.css', [], MRP_CACHE_BUSTER);
        wp_enqueue_script('meu-repositorio-admin-script', MRP_PLUGIN_URL . 'assets/js/admin-scripts.js', [], MRP_CACHE_BUSTER, true);

        wp_localize_script('meu-repositorio-admin-script', 'mrp_ajax', [
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('meu-repositorio-nonce'),
            'text' => [
                'updating' => __('Atualizando...', 'meu-repositorio-client'),
                'success' => __('Sucesso!', 'meu-repositorio-client'),
                'error' => __('Erro:', 'meu-repositorio-client'),
                'fatal_error' => __('Erro fatal.', 'meu-repositorio-client'),
                'force_update_running' => __('Forçando verificação...', 'meu-repositorio-client'),
                'force_update_success' => __('Verificação concluída!', 'meu-repositorio-client'),
                'saving' => __('Salvando...', 'meu-repositorio-client'),
                'activating' => __('Ativando...', 'meu-repositorio-client'),
            ]
        ]);
    }

    /**
     * Renderiza a página principal do repositório.
     */
    public function mrp_render_repo_page() {
        delete_transient('mrp_update_count');
        ?>
        <div class="wrap mrp-wrap">
            <div class="mrp-header-bar">
                <h1 class="mrp-header">
                    <span class="dashicons dashicons-download"></span>
                    <?php esc_html_e('Plugins Disponíveis', 'meu-repositorio-client'); ?>
                    <span class="mrp-badge version-up-to-date" style="font-size: 12px; vertical-align: middle; margin-left: 10px;"><?php echo esc_html($this->plugin_version); ?></span>
                </h1>
                <button id="mrp-force-update-button" class="button button-primary"><span class="dashicons dashicons-update"></span> <?php _e('Forçar Verificação', 'meu-repositorio-client'); ?></button>
            </div>
            
            <div class="mrp-card">
                <?php if (empty($this->api_url)): ?>
                    <div class="notice notice-warning is-dismissible"><p><?php printf(__('Por favor, %sconfigure a URL do repositório%s para começar.', 'meu-repositorio-client'), '<a href="' . esc_url(admin_url('admin.php?page=meu-repositorio-settings')) . '">', '</a>'); ?></p></div>
                <?php else: ?>
                    <?php 
                    $expires_at = get_option('mrp_token_expires_at');
                    $expiration_text = '';
                    if (!empty($expires_at)) {
                        $expiration_timestamp = strtotime($expires_at);
                        
                        if ($expiration_timestamp) {
                            $formatted_date = date_i18n(get_option('date_format'), $expiration_timestamp);
                            $expiration_text = sprintf(__(' Seu acesso é válido até <strong>%s</strong>.', 'meu-repositorio-client'), $formatted_date);
                            
                            $days_remaining = ceil(($expiration_timestamp - time()) / DAY_IN_SECONDS);
                            if ($days_remaining <= 7) {
                                 $expiration_text .= ' <a href="https://wpmasters.com.br/produto/plano-mensal-wp-masters-atualizacoes-automaticas/" target="_blank" style="color: #d63638; font-weight: bold;">' . __('Renovar Agora', 'meu-repositorio-client') . '</a>';
                            }
                        } else {
                             // Fallback for raw display
                             $expiration_text = sprintf(__(' Seu acesso é válido até %s.', 'meu-repositorio-client'), esc_html($expires_at));
                        }
                    }
                    ?>
                    <p class="mrp-intro"><?php echo __('A lista abaixo mostra os plugins do seu repositório.', 'meu-repositorio-client') . $expiration_text; ?></p>
                    <?php
                    wp_update_plugins();
                    $folder_scanner = new MRP_FolderScanner($this->api_url);
                    $force_refresh = isset($_GET['force-check']);
                    $available_plugins = $folder_scanner->scan_for_plugins($force_refresh);
                    wp_clean_plugins_cache(); // Force cache clear to detect new installs
                    $installed_plugins = get_plugins();
                    if (is_wp_error($available_plugins) || empty($available_plugins)) {
                        $error_msg = is_wp_error($available_plugins) ? $available_plugins->get_error_message() : '';
                        
                        if (empty($available_plugins) && !is_wp_error($available_plugins)) {
                             // No plugins found, likely due to access restrictions
                             echo '<div class="notice notice-info inline"><p>' . 
                                  sprintf(__('Nenhum plugin disponível para o seu domínio (%s).', 'meu-repositorio-client'), parse_url(home_url(), PHP_URL_HOST)) . 
                                  ' ' . 
                                  __('Isso pode significar que você ainda não adquiriu nenhum produto ou seu plano não inclui acesso a downloads.', 'meu-repositorio-client') . 
                                  '</p><p><a href="https://wpmasters.com.br/loja" target="_blank" class="button button-primary">' . __('Adquirir Acesso / Falar com Suporte', 'meu-repositorio-client') . '</a></p></div>';
                        } else {
                             // Actual error
                             echo '<div class="notice notice-error"><p>' . esc_html($error_msg) . '</p></div>';
                        }
                    } else {
                        ?>
                        <table class="wp-list-table widefat fixed striped mrp-table">
                            <thead><tr><th scope="col"><?php _e('Plugin', 'meu-repositorio-client'); ?></th><th scope="col"><?php _e('Versão Instalada', 'meu-repositorio-client'); ?></th><th scope="col"><?php _e('Versão no Repositório', 'meu-repositorio-client'); ?></th><th scope="col" style="text-align:center;"><?php _e('Downloads Hoje', 'meu-repositorio-client'); ?></th><th scope="col" style="text-align:center;"><?php _e('Ação', 'meu-repositorio-client'); ?></th></tr></thead>
                            <tbody>
                            <?php foreach ($available_plugins as $plugin_slug => $plugin_data) {
                                $installed_version = __('Não Instalado', 'meu-repositorio-client');
                                $plugin_file_path = '';
                                $plugin_installed = false;
                                $is_active = false;

                                foreach ($installed_plugins as $path => $details) {
                                    $matches = false;
                                    $folder_name = dirname($path);
                                    
                                    // 1. Exact folder name match (Most reliable)
                                    if ($folder_name === $plugin_slug) {
                                        $matches = true;
                                    }
                                    // 2. Exact Name match
                                    elseif ($details['Name'] === $plugin_data['name']) {
                                        $matches = true;
                                    }
                                    // 3. TextDomain matches slug
                                    elseif (isset($details['TextDomain']) && $details['TextDomain'] === $plugin_slug) {
                                        $matches = true;
                                    }
                                    // 4. Path contains slug (Fallback for renamed folders like 'slug-master')
                                    elseif (strpos($path, $plugin_slug . '/') !== false || strpos($path, '/' . $plugin_slug . '.php') !== false) {
                                        $matches = true;
                                    }

                                    if ($matches) {
                                        $installed_version = $details['Version'];
                                        $plugin_file_path = $path;
                                        $plugin_installed = true;
                                        $is_active = is_plugin_active($path);
                                        break;
                                    }
                                }
                                
                                $is_update_available = $plugin_installed && version_compare($plugin_data['version'], $installed_version, '>');
                                
                                // Download limits
                                $limits = isset($plugin_data['download_limits']) ? $plugin_data['download_limits'] : null;
                                $can_download = $limits && isset($limits['can_download']) ? $limits['can_download'] : true;
                                $plugin_site_used = $limits ? $limits['plugin_site_used'] : 0;
                                $plugin_site_limit = $limits ? $limits['plugin_site_limit'] : 5;
                                $token_used = $limits ? $limits['token_used'] : 0;
                                $token_limit = $limits ? $limits['token_limit'] : 50;
                                ?>
                                <tr>
                                    <td><strong><?php echo esc_html($plugin_data['name']); ?></strong></td>
                                    <td><span class="mrp-badge <?php echo $plugin_installed ? ($is_update_available ? 'version-outdated' : 'version-up-to-date') : 'version-unknown'; ?>"><?php echo esc_html($installed_version); ?></span></td>
                                    <td><span class="mrp-badge version-available"><?php echo esc_html($plugin_data['version']); ?></span></td>
                                    <td style="text-align:center;">
                                        <?php if ($limits): ?>
                                            <div style="font-size: 12px;">
                                                <div style="margin-bottom: 3px;">
                                                    <span class="dashicons dashicons-download" style="font-size: 14px; color: <?php echo $can_download ? '#2271b1' : '#d63638'; ?>; vertical-align: middle;"></span>
                                                    <strong style="color: <?php echo $can_download ? '#2271b1' : '#d63638'; ?>;">
                                                        <?php echo $plugin_site_used; ?>/<?php echo $plugin_site_limit; ?>
                                                    </strong>
                                                    <span style="color: #646970;">(plugin)</span>
                                                </div>
                                                <div>
                                                    <span class="dashicons dashicons-admin-network" style="font-size: 14px; color: <?php echo $can_download ? '#2271b1' : '#d63638'; ?>; vertical-align: middle;"></span>
                                                    <strong style="color: <?php echo $can_download ? '#2271b1' : '#d63638'; ?>;">
                                                        <?php echo $token_used; ?>/<?php echo $token_limit; ?>
                                                    </strong>
                                                    <span style="color: #646970;">(token)</span>
                                                </div>
                                            </div>
                                        <?php else: ?>
                                            <span style="color: #646970;">—</span>
                                        <?php endif; ?>
                                    </td>
                                    <td style="text-align:center;">
                                        <?php if ($plugin_installed): ?>
                                            <?php if ($is_update_available): ?>
                                                <button class="button button-primary plugin-action-button" 
                                                        data-state="update" 
                                                        data-plugin-file="<?php echo esc_attr($plugin_file_path); ?>" 
                                                        data-plugin-url="<?php echo esc_url($plugin_data['path']); ?>"
                                                        <?php echo !$can_download ? 'disabled title="Limite de downloads diário atingido"' : ''; ?>>
                                                    <?php _e('Atualizar Agora', 'meu-repositorio-client'); ?>
                                                </button>
                                                <?php if (!$can_download): ?>
                                                    <br><small style="color: #d63638;"><?php _e('Limite atingido', 'meu-repositorio-client'); ?></small>
                                                <?php endif; ?>
                                            <?php elseif (!$is_active): ?>
                                                <button class="button button-secondary plugin-action-button" data-state="activate" data-plugin-file="<?php echo esc_attr($plugin_file_path); ?>"><?php _e('Ativar', 'meu-repositorio-client'); ?></button>
                                            <?php else: ?>
                                                <button class="button" disabled><span class="dashicons dashicons-yes"></span> <?php _e('Ativo', 'meu-repositorio-client'); ?></button>
                                            <?php endif; ?>
                                        <?php else: ?>
                                            <button class="button button-primary plugin-action-button" 
                                                    data-state="install" 
                                                    data-plugin-file="" 
                                                    data-plugin-url="<?php echo esc_url($plugin_data['path']); ?>"
                                                    <?php echo !$can_download ? 'disabled title="Limite de downloads diário atingido"' : ''; ?>>
                                                <?php _e('Instalar', 'meu-repositorio-client'); ?>
                                            </button>
                                            <?php if (!$can_download): ?>
                                                <br><small style="color: #d63638;"><?php _e('Limite atingido', 'meu-repositorio-client'); ?></small>
                                            <?php endif; ?>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php } ?>
                            </tbody>
                        </table>
                    <?php } ?>
                <?php endif; ?>
            </div>
            
            <p class="mrp-footer-ad" style="text-align: right; font-size: 11px; margin-top: 15px; color: #a0a0a0;">
                <?php _e('Impulsionado por', 'meu-repositorio-client'); ?> <a href="https://wpmasters.com.br" target="_blank" style="color: #a0a0a0; text-decoration: none; font-weight: 500;">WP Masters</a>
            </p>
            </div>
        <?php
    }

    /**
     * Verifica as atualizações disponíveis no repositório.
     */
    public function mrp_check_for_updates($transient) {
        if (empty($transient->checked) || empty($this->api_url)) {
            return $transient;
        }
        delete_transient('mrp_update_count');
        $folder_scanner = new MRP_FolderScanner($this->api_url);
        $available_plugins = $folder_scanner->scan_for_plugins();
        if (is_wp_error($available_plugins) || empty($available_plugins)) {
            return $transient;
        }
        $installed_plugins = get_plugins();
        foreach ($available_plugins as $plugin_slug => $plugin_data) {
            $plugin_file_path = false;
            foreach ($installed_plugins as $path => $details) {
                if (strpos($path, $plugin_slug . '/') === 0 || $details['Name'] === $plugin_data['name']) {
                    $plugin_file_path = $path;
                    break;
                }
            }
            if ($plugin_file_path && isset($installed_plugins[$plugin_file_path]) && version_compare($plugin_data['version'], $installed_plugins[$plugin_file_path]['Version'], '>')) {
                $update_info = new \stdClass();
                $update_info->slug = $plugin_slug;
                $update_info->plugin = $plugin_file_path;
                $update_info->new_version = $plugin_data['version'];
                $update_info->package = $plugin_data['path'];
                $transient->response[$plugin_file_path] = $update_info;
            }
        }
        return $transient;
    }

    /**
     * Manipula a requisição AJAX para ativar um plugin.
     */
    public function mrp_handle_activate_plugin() {
        check_ajax_referer('meu-repositorio-nonce', 'security');
        if (!current_user_can('activate_plugins')) {
            wp_send_json_error(['message' => __('Permissão insuficiente.', 'meu-repositorio-client')]);
        }
        $plugin_file = isset($_POST['plugin_file']) ? sanitize_text_field($_POST['plugin_file']) : '';
        if (empty($plugin_file)) {
            wp_send_json_error(['message' => __('Arquivo do plugin não especificado.', 'meu-repositorio-client')]);
        }
        
        if (!function_exists('activate_plugin')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        $result = activate_plugin($plugin_file);
        if (is_wp_error($result)) {
            wp_send_json_error(['message' => $result->get_error_message()]);
        }
        wp_send_json_success(['message' => __('Plugin ativado com sucesso!', 'meu-repositorio-client')]);
    }

    /**
     * Manipula a requisição AJAX para atualizar um plugin.
     */
    public function mrp_handle_update_plugin() {
        check_ajax_referer('meu-repositorio-nonce', 'security');
        if (!current_user_can('update_plugins')) {
            wp_send_json_error(['message' => __('Permissão insuficiente.', 'meu-repositorio-client')]);
        }
        $plugin_file = isset($_POST['plugin_file']) ? sanitize_text_field($_POST['plugin_file']) : '';
        $package_url = isset($_POST['plugin_url']) ? esc_url_raw($_POST['plugin_url']) : '';
        
        // Allow empty plugin_file for installations, but package_url is mandatory.
        if (empty($package_url)) {
            wp_send_json_error(['message' => __('URL do pacote não fornecida.', 'meu-repositorio-client')]);
        }
        $plugin_updater = new PluginUpdater();
        $result = $plugin_updater->update_plugin($plugin_file, $package_url);
        if (is_wp_error($result)) {
            wp_send_json_error(['message' => $result->get_error_message()]);
        }
        
        // Invalida cache para forçar atualização dos limites de download
        $cache = new MRP_API_Cache();
        $cache_key = 'plugins_list_' . md5($this->api_url);
        $cache->invalidate($cache_key);
        
        delete_transient('mrp_update_count');
        wp_send_json_success(['message' => __('Plugin atualizado com sucesso!', 'meu-repositorio-client')]);
    }

    /**
     * Executa a rotina de atualização automática (usado pelo cron).
     */
    public function mrp_run_automatic_updates() {
        // Carrega a URL da API aqui para garantir que a função seja autossuficiente.
        $api_url = get_option('meu_repositorio_api_url', '');

        if (empty($api_url)) {
            echo "URL da API não configurada. Encerrando.\n";
            return;
        }
        
        echo "Acessando o repositório em: {$api_url}\n";
        
        $folder_scanner = new MRP_FolderScanner($api_url);
        $available_plugins = $folder_scanner->scan_for_plugins();

        if (is_wp_error($available_plugins)) {
            echo "Erro ao buscar plugins: " . $available_plugins->get_error_message() . "\n";
            return;
        }

        if (empty($available_plugins)) {
            echo "Nenhum plugin encontrado no repositório.\n";
            return;
        }

        if (!function_exists('get_plugins')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        $installed_plugins = get_plugins();
        $plugin_updater = new PluginUpdater();
        $updates_found = 0;

        foreach ($available_plugins as $plugin_slug => $plugin_data) {
            foreach ($installed_plugins as $path => $details) {
                if ((strpos($path, $plugin_slug . '/') === 0 || $details['Name'] === $plugin_data['name']) && version_compare($plugin_data['version'], $details['Version'], '>')) {
                    echo "Atualização encontrada para {$plugin_data['name']}: {$details['Version']} -> {$plugin_data['version']}. Iniciando atualização...\n";
                    $result = $plugin_updater->update_plugin($path, $plugin_data['path']);
                    if(is_wp_error($result)) {
                         echo "ERRO ao atualizar {$plugin_data['name']}: " . $result->get_error_message() . "\n";
                    } else {
                         echo "{$plugin_data['name']} atualizado com sucesso!\n";
                         $updates_found++;
                    }
                    break;
                }
            }
        }
        
        if($updates_found === 0) {
            echo "Nenhuma atualização necessária. Todos os plugins estão na versão mais recente.\n";
        }

        delete_transient('mrp_update_count');
    }

    /**
     * Limpa o cache de transientes para forçar uma nova verificação.
     */
    public function mrp_force_update_plugins_ajax_handler() {
        check_ajax_referer('meu-repositorio-nonce', 'security');
        if (!current_user_can('update_plugins')) {
            wp_send_json_error(['message' => __('Permissão insuficiente.', 'meu-repositorio-client')]);
        }
        
        // 1. Limpa cache do WordPress
        delete_site_transient('update_plugins');
        
        // 2. Limpa cache da nossa API
        $cache = new MRP_API_Cache();
        $cache->clear_all();
        
        wp_send_json_success(['message' => __('Verificação forçada. Cache limpo.', 'meu-repositorio-client')]);
    }

    /**
     * Ações na desativação do plugin.
     */
    public static function mrp_deactivate() {
        delete_site_transient('update_plugins');
        delete_transient('mrp_update_count');
    }
}

/**
 * Função global para iniciar o plugin no gancho correto.
 */
function mrp_run_plugin() {
    MeuRepositorioPlugin::get_instance();
}
add_action('plugins_loaded', 'mrp_run_plugin');

// Registra o hook de desativação
register_deactivation_hook(__FILE__, ['MeuRepositorioPlugin', 'mrp_deactivate']);