<?php
/**
 * Classe para escanear um repositório de plugins via JSON API.
 * @version 1.9 - Adicionado envio da URL do site cliente para rastreamento.
 */
class MRP_FolderScanner {
    
    /**
     * URL da API do repositório.
     * @var string
     */
    private $api_url;
    
    /**
     * Construtor da classe.
     * @param string $url URL do endpoint JSON.
     */
    public function __construct($url) {
        $this->api_url = $url;
    }
    
    /**
     * Escaneia o repositório em busca de plugins válidos.
     * @param bool $force_refresh Forçar atualização ignorando cache
     * @return array|WP_Error Array de plugins ou um objeto WP_Error em caso de falha.
     */
    public function scan_for_plugins($force_refresh = false) {
        if (empty($this->api_url)) {
            return new WP_Error('api_not_configured', 'A URL da API do repositório não está configurada.');
        }

        // Usa cache se disponível
        $cache = new MRP_API_Cache();
        $cache_key = 'plugins_list_' . md5($this->api_url);
        
        return $cache->get_or_fetch($cache_key, function() {
            return $this->fetch_from_api();
        }, $force_refresh);
    }
    
    /**
     * Busca dados diretamente da API (método privado usado pelo cache)
     */
    private function fetch_from_api() {

        $plugins = [];
        
        // --- MODIFICAÇÃO PRINCIPAL ---
        // Adiciona a URL do site cliente como um parâmetro na requisição da API.
        // Isto é essencial para o servidor rastrear qual site está usando o token.
        $args = ['requesting_site_url' => home_url()];
        
        // CORRIGIDO: Envia como client_token para não sobrescrever o token da API
        $client_token = get_option('meu_repositorio_license_token', '');
        if (!empty($client_token)) {
            $args['client_token'] = $client_token;
        }

        $api_url_with_tracking = add_query_arg($args, $this->api_url);
        // --- FIM DA MODIFICAÇÃO ---
        
        // Aumenta o timeout e usa a nova URL com rastreamento.
        $response = wp_remote_get($api_url_with_tracking, ['timeout' => 25, 'sslverify' => false]);
        
        if (is_wp_error($response)) {
            return new WP_Error('api_error', 'Erro ao acessar o repositório JSON: ' . $response->get_error_message());
        }
        
        $body = wp_remote_retrieve_body($response);
        if (empty($body)) {
            return new WP_Error('api_empty', 'A resposta do repositório JSON está vazia.');
        }

        $data = json_decode($body);
        if (json_last_error() !== JSON_ERROR_NONE) {
            return new WP_Error('json_error', 'Erro ao decodificar o JSON: ' . json_last_error_msg());
        }

        // Check if the server returned a structured error (e.g., 403 Forbidden, 400 Bad Request)
        if (is_object($data) && (isset($data->code) || isset($data->message))) {
            $code = isset($data->code) ? $data->code : 'api_error';
            $message = isset($data->message) ? $data->message : __('Erro desconhecido na API.', 'meu-repositorio-client');
            
            // Se houver dados adicionais (data.status, etc), podemos incluir, mas a mensagem é o principal.
            return new WP_Error($code, $message);
        }

        if (!is_array($data)) {
            return new WP_Error('data_format_error', 'O formato do JSON retornado não é um array de plugins. Resposta: ' . substr(print_r($data, true), 0, 200));
        }

        // Logic to save expiration moved inside the loop to capture body data, 
        // but we need to handle the case where loop doesn't run or finds nothing.
        // We will finalize the option update after the loop.

        // Capture Token Expiration from Body (Preferred) or Header (Fallback)
        $expires_at_header = wp_remote_retrieve_header($response, 'X-MRP-Token-Expires');
        
        // Debug Logging
        $debug_token = wp_remote_retrieve_header($response, 'X-MRP-Debug-Token-Received');
        $debug_expires = wp_remote_retrieve_header($response, 'X-MRP-Debug-Expires-Found');
        if ($debug_token || $debug_expires) {
            error_log("[MRP Client] Debug - Token: $debug_token, Expires: $debug_expires");
        }

        $expires_at_body = null;

        foreach ($data as $plugin_item) {
            // Try to find expiration in the first item (only once)
            if (empty($expires_at_body) && isset($plugin_item->token_expires_at)) {
                $expires_at_body = $plugin_item->token_expires_at;
            }
            
            // Verifica se todos os campos necessários existem.
            $has_title = isset($plugin_item->title->rendered);
            $has_slug = isset($plugin_item->slug);
            $has_version = isset($plugin_item->meta->mtf_versao);
            $has_url = isset($plugin_item->meta->mtf_url);
            
            if ($has_title && $has_slug && $has_version && $has_url) {
                
                $plugin_slug = $plugin_item->slug;

                $plugins[$plugin_slug] = [
                    'name'    => $plugin_item->title->rendered,
                    'version' => $plugin_item->meta->mtf_versao,
                    'path'    => $plugin_item->meta->mtf_url,
                    'type'    => 'zip',
                    'slug'    => $plugin_slug,
                    'download_limits' => isset($plugin_item->download_limits) ? (array) $plugin_item->download_limits : null,
                    'visibility' => isset($plugin_item->visibility) ? $plugin_item->visibility : 'restrito'
                ];
            } else {
                // Log why it failed
                error_log(sprintf(
                    '[MRP Client] Plugin rejected: %s. Missing: %s%s%s%s',
                    $plugin_item->slug ?? 'unknown',
                    !$has_title ? 'title ' : '',
                    !$has_slug ? 'slug ' : '',
                    !$has_version ? 'version ' : '',
                    !$has_url ? 'url' : ''
                ));
            }
        }
        
        // Save Expiration Date
        $final_expires_at = $expires_at_body ? $expires_at_body : $expires_at_header;
        if (!empty($final_expires_at)) {
            update_option('mrp_token_expires_at', $final_expires_at);
        } else {
            // Only delete if we are sure we didn't get it (and it wasn't just missing from this specific request but valid previously? 
            // No, if the request succeeds and has no date, it means no expiration or lost access).
            delete_option('mrp_token_expires_at');
        }

        if (empty($plugins) && !empty($data)) {
            return new WP_Error('no_valid_plugins', 'A resposta JSON foi recebida, mas nenhum plugin com a estrutura correta (title, slug, meta.mtf_versao, meta.mtf_url) foi encontrado.');
        }
        
        return $plugins;
    }
}